```php
<?php

namespace App\Http\Controllers\web;

use App\Models\Plan;
use App\Models\Team;
use App\Models\User;
use Illuminate\Http\Request;
use App\Services\OrderService;
use App\Enums\OrderPaymentTypeEnum;
use Jenssegers\Agent\Facades\Agent;
use App\Http\Controllers\Controller;
use App\Http\Controllers\PaymentController;
use App\Http\Resources\api\v1\OrderPaymentResource;

class SubscriptionController extends Controller
{
    public function index()
    {
        if (Agent::isMobile())
            return view('_pages.mobile.subscriptions');
        return view('_pages.subscriptions');
    }

    public function purchase(Request $request)
    {
        $purchaser = $request->team() ? $request->team() : $request->user();

    if ($purchaser->hasValidSubscription()) {
            return back()->with('status', __('you already have active subscriptions'));
        }

        $subscription = Plan::findOrFail($request->plan);
        $orderData = $this->createSubscriptionOrder($subscription, $purchaser);

        return redirect()->action([PaymentController::class, 'create'], [
            'tracking_code' => $orderData['tracking_code'],
            'uuid' => $orderData['uuid'],
            'payment_method' => $request->payment_method,
        ]);
    }

    protected function createSubscriptionOrder(Plan $plan, Team|User $purchaser): array
    {
        $amount = $plan->base_price;

        if ($purchaser instanceof Team) {
            $amount += $this->calculateSubscriptionPrice($plan, $purchaser);
        }

        $order = OrderService::createOrder($purchaser, $plan, data: [
            'amount' => $amount
        ]);

        $resource = (new OrderPaymentResource($order))->toArray(request());

        return array_merge($resource, [
            'payment_type' => OrderPaymentTypeEnum::Payment
        ]);
    }

    protected function calculateSubscriptionPrice(Plan $plan, Team $team)
    {
        return ($team->users()->count() - 1) * $plan->additional_user_price;
    }
}

```
