<?php

namespace App\Http\Actions\Admin\Shop;

use Carbon\Carbon;
use App\Models\User;
use App\Models\Order;
use App\Models\Category;
use App\Models\Shop;
use Illuminate\View\View;

class SailOverviewAction
{
    /**
     * Handle the incoming dashboard request.
     */
    public function __invoke(): View
    {
        $year        = Carbon::now()->year;
        $months      = $this->farsiMonths();

        // سفارش‌ها
        $newOrdersCount       = $this->countTodayAwaiting();
        $todayOrdersCount     = Order::forShop(shopId: auth()->user()->shop->id)->today()->count();
        $thisMonthOrdersCount = Order::forShop(shopId: auth()->user()->shop->id)->thisMonth()->count();
        $totalOrdersCount     = Order::forShop(shopId: auth()->user()->shop->id)->count();
        $abandonedOrdersCount = Order::forShop(shopId: auth()->user()->shop->id)->awaitingPayment()->count();

        // فروش‌ها
        $todaySales   = Order::forShop(shopId: auth()->user()->shop->id)->today()->completed()->sum('amount');
        $monthSales   = Order::forShop(shopId: auth()->user()->shop->id)->thisMonth()->completed()->sum('amount');
        $totalSales   = Order::forShop(shopId: auth()->user()->shop->id)->completed()->sum('amount');

        $monthlySales = $this->getYearlySalesByMonth($year);

        return view('pages.admin.shop.overview-sale', compact(
            'newOrdersCount',
            'todayOrdersCount',
            'thisMonthOrdersCount',
            'totalOrdersCount',
            'abandonedOrdersCount',
            'todaySales',
            'monthSales',
            'totalSales',
            'monthlySales',
            'months',
        ));
    }

    /**
     * Count orders created today with "AwaitingPayment" status.
     */
    private function countTodayAwaiting(): int
    {
        return Order::forShop(shopId: auth()->user()->shop->id)->today()
            ->awaitingPayment()
            ->count();
    }

    /**
     * Get an array of monthly sales totals for the given year.
     *
     * @return int[] [12 values indexed 0–11]
     */
    private function getYearlySalesByMonth(int $year): array
    {
        $salesByMonth = Order::forShop(shopId: auth()->user()->shop->id)->selectRaw('MONTH(created_at) as month, SUM(amount) as total')
            ->whereYear('created_at', $year)
            ->groupBy('month')
            ->orderBy('month')
            ->pluck('total', 'month');

        // ensure all 12 months are present
        return collect(range(1, 12))
            ->map(fn($m) => (int) ($salesByMonth->get($m) ?: 0))
            ->all();
    }

    /**
     * Persian month names.
     *
     * @return string[]
     */
    private function farsiMonths(): array
    {
        return [
            'فروردین',
            'اردیبهشت',
            'خرداد',
            'تیر',
            'مرداد',
            'شهریور',
            'مهر',
            'آبان',
            'آذر',
            'دی',
            'بهمن',
            'اسفند',
        ];
    }
}
