<?php

namespace App\Http\Controllers\Admin;

use Carbon\Carbon;
use App\Models\Product;
use App\Models\Category;
use App\Services\SweetAlert;
use Illuminate\Http\Request;
use App\Cache\CategoryTreeCache;
use App\Enums\ProductStatusEnum;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Route;
use App\Services\Models\ProductService;
use App\Services\Models\CategoryService;
use App\Http\Requests\Admin\ProductRequest;

class ProductController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Product::with('category' , 'attribute');

        if (request()->has('q')) {

            $query->where(function ($query) {
                $query->where('title', 'like', '%' . request('q') . '%');
                $query->orWhere('code', request('q'));
            });
        }
        if (request()->filled('category_id')) {
            $query->where('category_id', request('category_id'));
        }

        if (request()->filled('category')) {
            $children = CategoryService::getChildrenById(cacheable(CategoryTreeCache::class)->get(), $request->category);
            $ids = collect($children)->pluck('id');
            $ids->push($request->category);
            $query->whereIn('category_id', $ids);
        }


        if (Route::is('admin.shop.product-selection')) {
            $query->where('status', ProductStatusEnum::Published);
        }

        // Price range
        if ($request->filled('price_min')) {
            $query->where('price', '>=', $request->input('price_min'));
        }
        if ($request->filled('price_max')) {
            $query->where('price', '<=', $request->input('price_max'));
        }

        if ($request->filled('tag')) {
            $query->whereHas('tags', function ($q) use ($request) {
                $q->where('id', $request->input('tag'));
            });
        }

        if ($request->filled('second_hand')) {
            $query->where('second_hand', $request->input('second_hand'));
        }
        if ($request->filled('created_at')) {
            $date = Carbon::parse($request->input('created_at'));
            $query->whereBetween('created_at', [
                $date->copy()->startOfDay(),
                $date->copy()->endOfDay()
            ]);
        }

        $products = $query->paginate(20);


        $shopInventories = [];
        if (Route::is('admin.shop.product-selection')) {
            $shopInventories = auth()->user()->shop->inventories
                ->pluck('product_id')
                ->toArray();
        }

        return view('pages.admin.product.list', compact('products', 'shopInventories'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {

        if (isset($request->category_id) && Category::where('id', $request->category_id)->exists()) {
            $category = Category::find($request->category_id);
        } else {
            return redirect()->route('admin.categories.index')->with('sweetalert', SweetAlert::error(__('error'), __('before creating a product you need to select a category')));
        }

        return view('pages.admin.product.create', compact('category'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(ProductRequest $request)
    {

        $product = (new ProductService())->createOrUpdate($request->validated());

        return redirect()->route('admin.product.index')->with('sweetalert', SweetAlert::success(__('success'), __('product created successfully')));
    }

    /**
     * Display the specified resource.
     */
    public function show(Product $product)
    {
        return view('pages.admin.product.show', compact('product'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Request $request, Product $product)
    {
        if (isset($product->category_id) && Category::where('id', $product->category_id)->exists()) {
            $category = Category::find(id: $product->category_id);
        } else {
            return redirect()->route('admin.product.show', $product->id)->with('sweetalert', SweetAlert::error(__('error'), __('there was problem with the edit')));
        }

        // dd($product->attribute->keyBy('category_attribute_id'));
        return view('pages.admin.product.edit', compact('category', 'product'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(ProductRequest $request, Product $product)
    {
        $result = (new ProductService())->createOrUpdate($request->validated(), $product);

        return redirect()->route('admin.product.show', $product->id)->with('sweetalert', SweetAlert::success(__('success'), __('product created successfully')));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Category $category, Product $product)
    {
        $product->delete();

        return redirect()->route('admin.product.index')->with('sweetalert', SweetAlert::success(__('success'), __('product deleted successfully')));
    }
}
