<?php

namespace App\Imports;

use App\Enums\ProductStatusEnum;
use App\Models\Product;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;

class ProductImport implements ToModel, WithHeadingRow
{
    /**
     * @param array $row
     *
     * @return \Illuminate\Database\Eloquent\Model|null
     */
    public function model(array $row)
    {


        if ($this->toBool($row['is_duplicated'])) {
            return null;
        }

        $product = new Product([
            'id'                 => $row['id'],
            'user_id'            => 1,
            'category_id'        => $row['category_id'],
            'title'              => $row['title'],
            'code'               => $row['code'],
            'slug'               => Product::generateSlug($row['title']),
            'short_description'  => $this->cleanText(substr($row['short_description'], 0, 230) . "..."),
            'content'            => $this->cleanText($row['short_description']),
            'inventory'          => $row['inventory'],
            'brand_img'          => url('uploads/' . $row['code'] . '-1.jpg'),
            'price'              => $this->cleanPrice($row['price']),
            'weight'             => $row['weight'],
            'second_hand'        => $this->toBool($row['second_hand']),
            'status'             => ProductStatusEnum::Published,
            'is_duplicated'      => $this->toBool($row['is_duplicated']),
        ]);

        $product->save(); // Save first so we can attach images

        $this->attachImages($product);

        return $product;
    }


    private function cleanPrice($value)
    {
        return (int)str_replace(',', '', $value);
    }

    private function toBool($value)
    {
        return in_array(strtolower(trim($value)), ['1', 'true', 'yes', 'بله']);
    }
    private function cleanText($value)
    {
        return mb_convert_encoding($value, 'UTF-8', 'UTF-8');
    }
    private function attachImages(Product $product)
    {
        $code = $product->code;
        $directory = public_path('uploads');

        $supportedExtensions = ['jpg', 'jpeg', 'png', 'webp'];

        foreach ($supportedExtensions as $ext) {
            // Check for files like 1002-1.jpg, 1002-2.png, etc.
            foreach (glob("$directory/{$code}-*.$ext") as $filePath) {
                $filename = basename($filePath);
                
                $product->update(['brand_img' => asset("uploads/{$filename}")]);
                // Save path in DB (e.g., with ProductImage model)
                $product->metas()->create([
                    'key' => 'gallery',
                    'value' => asset("uploads/{$filename}"),
                    'data' => json_encode([$filePath]),
                ]);
            }
        }
    }
}
