<?php

namespace App\Models;

use App\Traits\HasWarehouse;
use App\Traits\HasUniqueSlug;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasManyThrough;

class Shop extends Model
{
    use HasUniqueSlug, HasWarehouse;
    protected string $slugSource = 'name';
    protected $fillable = [
        'user_id',
        'name',
        'slug',
        'logo',
        'description',
        'phone',
        'email',
        'address',
        'city',
        'state',
        'postal_code',
        'country',
        'is_approved',
        'approved_at',
        'consumer_key',
        'consumer_secret',
        'woo_url',
    ];

    public function user(): BelongsTo
    {
        return $this->belongsTo(related: User::class);
    }

    public function inventories(): HasMany
    {
        return $this->hasMany(related: ProductInventory::class);
    }
    public function inventoryVariants(): HasManyThrough
    {
        return $this->hasManyThrough(related: ProductInventoryVariant::class, through: ProductInventory::class, firstKey: 'shop_id', secondKey: 'product_inventory_id');
    }
    public function warehouses(): HasMany
    {
        return $this->hasMany(related: Warehouse::class);
    }
    public function quotes()
    {
        return $this->hasMany(Quote::class, 'supplier_id');
    }

    public function hasOneWarehouse(): bool
    {
        return $this->warehouses()->count() >= 1;
    }

    // scope
    public function scopeApproved($query)
    {
        return $query->where('is_approved', true);
    }
    // scope
    public function scopeNotApproved($query)
    {
        return $query->where('is_approved', false);
    }

    //helpers
    public function isShopWooCommerceCredentialExist(): bool
    {
        if ($this->consumer_key === null || $this->consumer_secret === null || $this->woo_url === null) {
            return false;
        }
        return true;
    }

    public function acceptedQuotes(string $approach = 'join')
    {
        if ($approach === 'sub-query') {
            return $this->quotes()
                ->whereIn('id', function ($query) {
                    $query->select('selected_quote_id')
                        ->from('pre_invoices')
                        ->whereNotNull('selected_quote_id');
                })
                ->get();
        }

        if ($approach === 'join') {
            return Quote::query()
                ->join('pre_invoices', 'pre_invoices.selected_quote_id', '=', 'quotes.id')
                ->where('quotes.supplier_id', $this->id)
                ->select([
                    'quotes.*',
                    'pre_invoices.id as pre_invoice_id',
                    'pre_invoices.customer_name',
                    'pre_invoices.company_name',
                    'pre_invoices.email',
                    'pre_invoices.phone',
                    'pre_invoices.city',
                    'pre_invoices.postal_code',
                    'pre_invoices.status',
                ])
                ->get();
        }

        return collect([]);
    }
}
