<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;

use App\Enums\WalletTypeEnum;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Spatie\Permission\Traits\HasPermissions;
use Spatie\Permission\Traits\HasRoles;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, HasPermissions, HasRoles, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'username',
        'shop_id',
        'email',
        'phone',
        'password',
        'admin_layout',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'phone_verified_at' => 'datetime',
            'password' => 'hashed',
        ];
    }

    public function preInvoices(): HasMany
    {
        return $this->hasMany(related: PreInvoice::class , foreignKey: 'creator_id');
    }
    public function addresses(): HasMany
    {
        return $this->hasMany(Address::class);
    }
    public function wallets(): HasMany
    {
        return $this->hasMany(Wallet::class);
    }
    public function wallet(): HasOne
    {
        return $this->hasOne(Wallet::class)->where('type', WalletTypeEnum::Toman);
    }
    public function shop(): HasOne
    {
        return $this->hasOne(Shop::class);
    }
    public function orders()
    {
        return $this->morphMany(Order::class, 'ordered_by');
    }

    public function isAdmin()
    {
        return $this->hasRole('Administrator') || $this->hasRole('Admin');
    }
    public function isNotAdmin()
    {
        return !$this->hasRole('Administrator') && !$this->hasRole('Admin');
    }
    public function isSuperAdmin()
    {
        return $this->hasRole('Administrator');
    }
    public function isShop()
    {
        return $this->hasRole('Shop');
    }

    public function hasShop(): bool
    {
        return $this->shop()->exists();
    }
    public function shopIsApproved(): bool
    {
        return $this->shop?->is_approved ?? false;
    }
    public function adminLayout(): string
    {
        $layout = $this->admin_layout == 'horizontal' ? true : false;
        return $layout ? 'horizontal' : 'vertical';
    }
}
