<?php

/**
 * Simple PHP Documentation Generator
 * Author: Alireza Hadizadeh
 */

declare(strict_types=1);

require_once __FILE__; // (self-contained)

// === Step 1: Setup ===
$root = __DIR__;
$outputDir = $root . '/docs';
$outputFile = $outputDir . '/doc-generator.json';

// Create the directory if it doesn't exist
@mkdir($outputDir, 0777, true);

$generator = new DocGenerator($root);
$result = $generator->generate();

file_put_contents(
    filename: $outputFile,
    data: json_encode(value: $result, flags: JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE)
);
echo "✅ Generated: {$outputFile}\n";

// === Step 2: Classes ===

class DocGenerator
{
    public function __construct(
        public string $root
    ) {}

    public function generate(): array
    {
        $project = new ProjectInfo($this->root);
        $composer = new ComposerInspector($this->root);
        $npm = new NpmInspector($this->root);
        $routes = new LaravelRoutes($this->root);
        $database = new DatabaseInspector($this->root);



        $dbData = $database->inspect();

        $diagram = new DatabaseDiagram($dbData);
        $diagram->saveFiles($this->root . '/docs');

        return [
            'meta' => $project->info(),
            'packages' => [
                'composer' => $composer->getPackages(),
                'npm' => $npm->getPackages(),
            ],
            'routes' => $routes->get(),
            'database' => $dbData,
        ];
    }
}


// === Domain Classes ===

class ProjectInfo
{
    public function __construct(
        private string $root
    ) {}

    public function info(): array
    {
        return [
            'generated_at' => date('c'),
            'php_version' => PHP_VERSION,
            'framework' => $this->detectFramework(),
            'framework_data' => $this->getFrameworkData(),
            'git_branch' => $this->shell('git rev-parse --abbrev-ref HEAD'),
            'commit' => $this->shell('git rev-parse HEAD'),
        ];
    }

    private function detectFramework(): string
    {
        if (file_exists($this->root . '/artisan') && is_dir($this->root . '/vendor/laravel/framework')) {
            return 'Laravel';
        }
        return 'Unknown';
    }

    private function getFrameworkData(): array
    {
        if ($this->detectFramework() !== 'Laravel') return [];

        $line = $this->shell('php artisan --version');
        if (preg_match('/Laravel Framework\s+([\d\.]+)/', $line, $m)) {
            $ver = $m[1];
        } else {
            $ver = 'unknown';
        }

        return [
            'name' => 'Laravel',
            'version' => $ver,
        ];
    }

    private function shell(string $cmd): ?string
    {
        if (!is_callable('shell_exec')) return null;
        return trim((string)@shell_exec($cmd . ' 2>/dev/null'));
    }
}


class ComposerInspector
{
    public function __construct(
        private string $root
    ) {}

    public function getPackages(): array
    {
        $file = $this->root . '/composer.lock';
        if (!file_exists($file)) return [];
        $data = json_decode(file_get_contents($file), true);
        $out = [];
        foreach ($data['packages'] ?? [] as $p) {
            $out[$p['name']] = $p['version'];
        }
        return $out;
    }
}


class NpmInspector
{
    public function __construct(
        private string $root
    ) {}

    public function getPackages(): array
    {
        $file = $this->root . '/package.json';
        if (!file_exists($file)) return [];
        $data = json_decode(file_get_contents($file), true);
        return ($data['dependencies'] ?? []) + ($data['devDependencies'] ?? []);
    }
}


class LaravelRoutes
{
    public function __construct(
        private string $root
    ) {}

    public function get(): array
    {
        if (!file_exists($this->root . '/artisan')) return ['_skipped' => 'no artisan'];
        $json = @shell_exec('php artisan route:list --json 2>/dev/null');
        if (!$json) return ['_skipped' => 'no route list'];
        return json_decode($json, true) ?: [];
    }
}

class DatabaseInspector
{
    private ?PDO $pdo = null;
    private array $env = [];

    public function __construct(private string $root)
    {
        $this->env = $this->parseDotEnv($root . '/.env');
        $this->pdo = $this->connect();
    }

    public function inspect(): array
    {
        if (!$this->pdo) {
            return ['_error' => 'Database connection unavailable'];
        }

        $driver = $this->pdo->getAttribute(PDO::ATTR_DRIVER_NAME);
        $result = ['driver' => $driver, 'tables' => []];

        try {
            $tables = $this->getTables($driver);
            foreach ($tables as $table) {
                $meta = [
                    'name'          => $table,
                    'columns'       => $this->getColumns($driver, $table),
                    'indexes'       => $this->getIndexes($driver, $table),
                    'foreign_keys'  => $this->getForeignKeys($driver, $table),
                ];
                $meta['sql_schemas'] = [
                    'native' => $this->getCreateStatement($driver, $table),
                    'sqlite' => $this->generateSQLSchema($meta, 'sqlite'),
                    'mysql'  => $this->generateSQLSchema($meta, 'mysql'),
                    'pgsql'  => $this->generateSQLSchema($meta, 'pgsql'),
                ];

                $result['tables'][$table] = $meta;
            }
        } catch (Throwable $e) {
            $result['_error'] = $e->getMessage();
        }

        return $result;
    }

    /* --------------------------------------------- */
    /* 🧠  Helpers                                   */
    /* --------------------------------------------- */

    private function connect(): ?PDO
    {
        try {
            $driver = strtolower($this->env['DB_CONNECTION'] ?? 'sqlite');

            switch ($driver) {
                case 'mysql':
                    $dsn = sprintf(
                        'mysql:host=%s;dbname=%s;charset=utf8mb4',
                        $this->env['DB_HOST'] ?? '127.0.0.1',
                        $this->env['DB_DATABASE'] ?? ''
                    );
                    return new PDO($dsn, $this->env['DB_USERNAME'] ?? 'root', $this->env['DB_PASSWORD'] ?? '', [
                        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    ]);

                case 'pgsql':
                case 'postgres':
                    $dsn = sprintf(
                        'pgsql:host=%s;dbname=%s',
                        $this->env['DB_HOST'] ?? '127.0.0.1',
                        $this->env['DB_DATABASE'] ?? ''
                    );
                    return new PDO($dsn, $this->env['DB_USERNAME'] ?? 'postgres', $this->env['DB_PASSWORD'] ?? '', [
                        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    ]);

                case 'sqlite':
                default:
                    $path = $this->env['DB_DATABASE'] ?? ($this->root . '/database/database.sqlite');
                    return new PDO('sqlite:' . $path, null, null, [
                        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    ]);
            }
        } catch (Throwable $e) {
            return null;
        }
    }

    private function getTables(string $driver): array
    {
        return match ($driver) {
            'sqlite' => $this->pdo->query("SELECT name FROM sqlite_master WHERE type='table' AND name NOT LIKE 'sqlite_%'")->fetchAll(PDO::FETCH_COLUMN),
            'mysql'  => $this->pdo->query("SHOW TABLES")->fetchAll(PDO::FETCH_COLUMN),
            'pgsql'  => $this->pdo->query("SELECT tablename FROM pg_tables WHERE schemaname='public'")->fetchAll(PDO::FETCH_COLUMN),
            default  => [],
        };
    }

    private function getColumns(string $driver, string $table): array
    {
        $columns = [];
        switch ($driver) {
            case 'sqlite':
                $stmt = $this->pdo->query("PRAGMA table_info('$table')");
                while ($col = $stmt->fetch(PDO::FETCH_ASSOC)) {
                    $columns[$col['name']] = [
                        'type'     => $col['type'],
                        'notnull'  => (bool) $col['notnull'],
                        'default'  => $col['dflt_value'],
                        'pk'       => (bool) $col['pk'],
                    ];
                }
                break;

            case 'mysql':
                $stmt = $this->pdo->query("SHOW COLUMNS FROM `$table`");
                while ($col = $stmt->fetch(PDO::FETCH_ASSOC)) {
                    $columns[$col['Field']] = [
                        'type'     => $col['Type'],
                        'null'     => $col['Null'],
                        'key'      => $col['Key'],
                        'default'  => $col['Default'],
                    ];
                }
                break;

            case 'pgsql':
                $stmt = $this->pdo->prepare("
                    SELECT column_name, data_type, is_nullable, column_default
                    FROM information_schema.columns
                    WHERE table_name = :t
                ");
                $stmt->execute(['t' => $table]);
                while ($col = $stmt->fetch(PDO::FETCH_ASSOC)) {
                    $columns[$col['column_name']] = [
                        'type'     => $col['data_type'],
                        'null'     => $col['is_nullable'],
                        'default'  => $col['column_default'],
                    ];
                }
                break;
        }
        return $columns;
    }

    private function getIndexes(string $driver, string $table): array
    {
        $indexes = [];
        switch ($driver) {
            case 'sqlite':
                $stmt = $this->pdo->query("PRAGMA index_list('$table')");
                $indexes = $stmt->fetchAll(PDO::FETCH_ASSOC);
                break;

            case 'mysql':
                $stmt = $this->pdo->query("SHOW INDEXES FROM `$table`");
                while ($idx = $stmt->fetch(PDO::FETCH_ASSOC)) {
                    $indexes[$idx['Key_name']][] = $idx['Column_name'];
                }
                break;

            case 'pgsql':
                $stmt = $this->pdo->prepare("
                    SELECT indexname, indexdef
                    FROM pg_indexes
                    WHERE tablename = :t
                ");
                $stmt->execute(['t' => $table]);
                $indexes = $stmt->fetchAll(PDO::FETCH_ASSOC);
                break;
        }
        return $indexes;
    }

    private function getForeignKeys(string $driver, string $table): array
    {
        $fks = [];
        switch ($driver) {
            case 'sqlite':
                $stmt = $this->pdo->query("PRAGMA foreign_key_list('$table')");
                $fks = $stmt->fetchAll(PDO::FETCH_ASSOC);
                break;

            case 'mysql':
                $stmt = $this->pdo->prepare("
                    SELECT
                        k.CONSTRAINT_NAME AS name,
                        k.COLUMN_NAME AS column_name,
                        k.REFERENCED_TABLE_NAME AS referenced_table,
                        k.REFERENCED_COLUMN_NAME AS referenced_column
                    FROM information_schema.KEY_COLUMN_USAGE k
                    WHERE
                        k.TABLE_SCHEMA = DATABASE()
                        AND k.TABLE_NAME = :t
                        AND k.REFERENCED_TABLE_NAME IS NOT NULL
                ");
                $stmt->execute(['t' => $table]);
                $fks = $stmt->fetchAll(PDO::FETCH_ASSOC);
                break;

            case 'pgsql':
                $stmt = $this->pdo->prepare("
                    SELECT
                        conname AS name,
                        conrelid::regclass AS table_name,
                        a.attname AS column_name,
                        confrelid::regclass AS ref_table,
                        af.attname AS ref_column
                    FROM pg_constraint c
                    JOIN pg_attribute a ON a.attnum = ANY(c.conkey) AND a.attrelid = c.conrelid
                    JOIN pg_attribute af ON af.attnum = ANY(c.confkey) AND af.attrelid = c.confrelid
                    WHERE c.contype = 'f' AND c.conrelid = :t::regclass
                ");
                $stmt->execute(['t' => $table]);
                $fks = $stmt->fetchAll(PDO::FETCH_ASSOC);
                break;
        }
        return $fks;
    }

    private function parseDotEnv(string $path): array
    {
        if (!file_exists($path)) return [];
        $lines = file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        $env = [];
        foreach ($lines as $line) {
            $line = trim($line);
            if ($line === '' || $line[0] === '#') continue;
            if (!str_contains($line, '=')) continue;
            [$key, $value] = explode('=', $line, 2);
            $key = trim($key);
            $value = trim($value);
            if ((str_starts_with($value, '"') && str_ends_with($value, '"')) ||
                (str_starts_with($value, "'") && str_ends_with($value, "'"))
            ) {
                $value = substr($value, 1, -1);
            }
            $env[$key] = $value;
        }
        return $env;
    }

    private function generateSQLSchema(array $tableMeta, string $dialect): string
    {
        $table = $tableMeta['name'] ?? 'unknown';
        $columns = $tableMeta['columns'] ?? [];
        $fks = $tableMeta['foreign_keys'] ?? [];
        $pks = [];

        $sql = "CREATE TABLE \"$table\" (\n";

        foreach ($columns as $colName => $col) {
            $line = "  \"$colName\" ";

            // map generic types
            $type = strtolower($col['type'] ?? 'text');
            $typeMap = [
                'sqlite' => ['int' => 'INTEGER', 'varchar' => 'TEXT', 'text' => 'TEXT', 'float' => 'REAL', 'double' => 'REAL'],
                'mysql'  => ['int' => 'INT', 'varchar' => 'VARCHAR(255)', 'text' => 'TEXT', 'float' => 'FLOAT', 'double' => 'DOUBLE'],
                'pgsql'  => ['int' => 'INTEGER', 'varchar' => 'VARCHAR(255)', 'text' => 'TEXT', 'float' => 'REAL', 'double' => 'DOUBLE PRECISION'],
            ];
            $matchType = 'TEXT';
            foreach ($typeMap[$dialect] as $k => $v) {
                if (str_contains($type, $k)) {
                    $matchType = $v;
                    break;
                }
            }
            $line .= $matchType;

            if (!empty($col['pk'])) {
                $pks[] = $colName;
            }

            if (!empty($col['notnull']) || ($col['null'] ?? '') === 'NO') {
                $line .= ' NOT NULL';
            }

            if (isset($col['default']) && $col['default'] !== null && $col['default'] !== '') {
                $def = trim($col['default'], "'");
                $line .= " DEFAULT '$def'";
            }

            $sql .= $line . ",\n";
        }

        // primary key
        if (!empty($pks)) {
            $sql .= "  PRIMARY KEY (" . implode(', ', array_map(fn($n) => "\"$n\"", $pks)) . "),\n";
        }

        // foreign keys
        foreach ($fks as $fk) {
            $col = $fk['column_name'] ?? $fk['from'] ?? null;
            $refTable = $fk['referenced_table'] ?? $fk['table'] ?? null;
            $refCol = $fk['referenced_column'] ?? $fk['to'] ?? 'id';
            if ($col && $refTable) {
                $sql .= "  FOREIGN KEY (\"$col\") REFERENCES \"$refTable\"(\"$refCol\"),\n";
            }
        }

        // remove last comma
        $sql = rtrim($sql, ",\n") . "\n);\n";

        return $sql;
    }

    private function getCreateStatement(string $driver, string $table): ?string
    {
        try {
            switch ($driver) {
                case 'sqlite':
                    $stmt = $this->pdo->prepare("SELECT sql FROM sqlite_master WHERE type='table' AND name = :t");
                    $stmt->execute(['t' => $table]);
                    return $stmt->fetchColumn();

                case 'mysql':
                    $stmt = $this->pdo->query("SHOW CREATE TABLE `$table`");
                    $row = $stmt->fetch(PDO::FETCH_ASSOC);
                    return $row['Create Table'] ?? null;

                case 'pgsql':
                    // PostgreSQL doesn't have SHOW CREATE TABLE, so use pg_dump if allowed
                    if (is_callable('shell_exec')) {
                        $db = $this->env['DB_DATABASE'] ?? '';
                        $user = $this->env['DB_USERNAME'] ?? 'postgres';
                        $cmd = sprintf("pg_dump --schema-only --no-owner --no-privileges -t %s %s 2>/dev/null", escapeshellarg($table), escapeshellarg($db));
                        $ddl = trim(shell_exec($cmd));
                        return $ddl ?: null;
                    } else {
                        // Fallback query (simplified approximation)
                        $stmt = $this->pdo->prepare("
                        SELECT 'CREATE TABLE ' || relname || E' (\n' ||
                            string_agg(
                                '    ' || a.attname || ' ' || pg_catalog.format_type(a.atttypid, a.atttypmod),
                                E',\n'
                            ) || E'\n);'
                        FROM pg_class c
                        JOIN pg_attribute a ON a.attrelid = c.oid
                        WHERE c.relname = :t AND a.attnum > 0 AND NOT a.attisdropped
                        GROUP BY c.relname;
                    ");
                        $stmt->execute(['t' => $table]);
                        return $stmt->fetchColumn();
                    }

                default:
                    return null;
            }
        } catch (Throwable $e) {
            return null;
        }
    }
}

class DatabaseDiagram
{
    public function __construct(private array $databaseSchema) {}

    /** Generate Mermaid ER diagram (v10.9+ compatible) */
    public function generate(): string
    {
        $out = "erDiagram\n";

        foreach ($this->databaseSchema['tables'] ?? [] as $tableName => $table) {
            // Ensure table names are alphanumeric
            $safeName = preg_replace('/[^A-Za-z0-9_]/', '_', $tableName);
            $out .= "    $safeName {\n";

            foreach ($table['columns'] ?? [] as $col => $meta) {
                $safeCol = preg_replace('/[^A-Za-z0-9_]/', '_', $col);
                $type = $this->normalizeType($meta['type'] ?? '');

                $key = '';
                if (!empty($meta['pk']) || ($meta['key'] ?? '') === 'PRI') {
                    $key = 'PK';
                } elseif (($meta['key'] ?? '') === 'MUL') {
                    $key =  '' ;//'IDX';
                } elseif (($meta['key'] ?? '') === 'UNI') {
                    $key = '' ;//'UNIQUE';
                }

                // Format: TYPE column_name KEY(optional)
                $out .= "        $type $safeCol $key\n";
            }

            $out .= "    }\n";
        }


        // Foreign keys as relationships
        foreach ($this->databaseSchema['tables'] ?? [] as $tableName => $table) {
            $from = preg_replace('/[^A-Za-z0-9_]/', '_', $tableName);
            foreach ($table['foreign_keys'] ?? [] as $fk) {
                $to = $fk['referenced_table'] ?? $fk['table'] ?? null;
                if ($to) {
                    $to = preg_replace('/[^A-Za-z0-9_]/', '_', $to);
                    $out .= "    $from }o--|| $to : FK\n";
                }
            }
        }

        return $out;
    }

    /** Save Mermaid + HTML viewer */
    public function saveFiles(string $outputDir): bool
    {
        @mkdir($outputDir, 0777, true);
        $mmdFile = rtrim($outputDir, '/') . '/schema.mmd';
        $htmlFile = rtrim($outputDir, '/') . '/schema.html';
        $diagram = $this->generate();

        // Save .mmd file
        file_put_contents($mmdFile, $diagram);

        // Build HTML viewer (Mermaid 10.9.4 compatible)
        $html = <<<HTML
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Database Diagram</title>
  <script type="module">
    import mermaid from "https://cdn.jsdelivr.net/npm/mermaid@10.9.4/dist/mermaid.esm.min.mjs";
    mermaid.initialize({ startOnLoad: true, theme: "neutral" });
  </script>
  <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-100 p-6">
  <h1 class="text-2xl font-bold mb-4">📊 Database ER Diagram</h1>
  <div class="bg-white rounded shadow p-4 overflow-auto">
    <pre class="mermaid text-sm whitespace-pre-wrap">$diagram</pre>
  </div>
</body>
</html>
HTML;

        file_put_contents($htmlFile, $html);

        echo "🧩 Diagram generated successfully:\n - $mmdFile\n - $htmlFile\n";
        return true;
    }

    private function normalizeType(string $raw): string
    {
        $raw = strtoupper($raw);

        // simplify compound types
        $map = [
            '/UNSIGNED/'   => 'INT',
            '/BIGINT|MEDIUMINT|SMALLINT|TINYINT/' => 'INT',
            '/INT/'        => 'INT',
            '/DECIMAL|DOUBLE|FLOAT|REAL|NUMERIC/' => 'FLOAT',
            '/CHAR|VARCHAR|TEXT|TINYTEXT|MEDIUMTEXT|LONGTEXT|ENUM|SET/' => 'STRING',
            '/DATE|DATETIME|TIMESTAMP|TIME|YEAR/' => 'DATETIME',
            '/JSON/'       => 'JSON',
            '/BOOL|BOOLEAN/' => 'BOOLEAN',
            '/BLOB|BINARY|VARBINARY/' => 'BINARY',
        ];

        foreach ($map as $pattern => $replacement) {
            if (preg_match($pattern, $raw)) {
                return $replacement;
            }
        }

        return 'STRING'; // default fallback
    }
}
